#ifndef __IControlRoot__
#define __IControlRoot__

#include <Basics/CSystemString.hpp>
#include <Basics/CCountedObject.hpp>
#include <MathTools/CDimension.hpp>
#include "../Windowing/IWindow.hpp"
#include "../Graphics/IImage.hpp"
#include "../Graphics/CGraphics.hpp"
#include "../Listeners/IDropFileListener.hpp"
#include "../Listeners/IMouseListener.hpp"
#include "../Listeners/IKeyboardListener.hpp"

//	===========================================================================

using Exponent::Basics::CSystemString;
using Exponent::MathTools::CDimension;
using Exponent::GUI::Windowing::IWindow;
using Exponent::GUI::Graphics::IImage;
using Exponent::GUI::Graphics::CGraphics;
using Exponent::Basics::CCountedObject;
using Exponent::GUI::Listeners::IDropFileListener;
using Exponent::GUI::Listeners::IKeyboardListener;
using Exponent::GUI::Listeners::IMouseListener;

//	===========================================================================

/**
 * @cond
 */
#ifndef __ICONTROL_GUARD__
#define __ICONTROL_GUARD__
	namespace Exponent { namespace GUI { namespace Controls { interface IControl; } } }
	using Exponent::GUI::Controls::IControl;
#endif
/**
 * @endcond
 */

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Controls
		{
			/**
			 * @interface IControlRoot IControlRoot.hpp
			 * @brief Interface to a control root
			 *
			 * @date 19/03/2005
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: IControlRoot.hpp,v 1.5 2007/02/11 02:16:22 paul Exp $
			 */
			interface IControlRoot : public IMouseListener, public IDropFileListener, public IKeyboardListener
			{
			public:

//	===========================================================================

				/**
				 * @enum EControlRootSerialiseOption
				 * @brief Options for serialising to disk
				 */
				enum EControlRootSerialiseOption
				{
					e_serialiseToDisk = 0,		/**< Serialise to disk */
					e_serialiseFromDisk			/**< Serialise from disk */
				};

//	===========================================================================

				/**
				 * Construction
				 */
				IControlRoot() { }

				/**
				 * Destruction
				 */
				virtual ~IControlRoot() { }

//	===========================================================================

				/**
				 * Lock a control (only this control recieves mouse info)
				 * @param control The control to lock
				 */
				virtual void lockControl(IControl *control) = 0;

				/**
				 * Unlock a control
				 */
				virtual void unlockControl() = 0;

				/**
				 * Is there a control locked
				 * @retval bool True if a control is locked, false otherwise
				 */
				virtual bool controlIsLocked() const = 0;

				/**
				 * Get the locked control
				 * @retval IControl* The locked control
				 */
				virtual IControl *getLockedControl() const = 0;

//	===========================================================================

				/**
				 * Set the parent window
				 */
				virtual void setParentWindow(IWindow *parent) = 0;

				/**
				 * Get the parent window
				 */
				virtual IWindow *getParentWindow() const = 0;

				/**
				 * Get window offset
				 */
				virtual CPoint getWindowOffset() = 0;

//	===========================================================================

				/**
				 * Get the control at this point (point is relative to this' top left)
				 * @param point The position relative to this roots top left
				 * @retval IControl* The control, or null on error
				 */
				virtual IControl *getControlAtPoint(const CPoint &point) = 0;

				/**
				 * Get the control at this point, checks for mouse enabling
				 * @param point The position relative to this roots top left
				 * @retval IControl* The control, or null on error
				 */
				virtual IControl *getMouseEnabledControlAtPoint(const CPoint &point) = 0;

				/**
				 * Get he control at this point, checks for drop file enabling
				 * @param point The position relative to this roots top left
				 * @retval IControl* The control, or null on error
				 */
				virtual IControl *getDropEnabledControlAtPoint(const CPoint &point) = 0;

				/**
				 * Get the key enabled control at this point, checks for key handling
				 * @param point The position relative to this roots top left
				 * @retval IControl* The control, or null on error
				 */
				virtual IControl *getKeyEnabledControlAtPoint(const CPoint &point) = 0;

//	===========================================================================

				/**
				 * Add a control
				 * @param control The control to add
				 */
				virtual void addControl(IControl *control) = 0;

				/**
				 * Remove a control
				 * @param control The control to remove
				 */
				virtual void removeControl(IControl *control) = 0;

				/**
				 * Remove all controls
				 */
				virtual void clearControls() = 0;

				/**
				 * Get a specific index, or with a specific id, if isIndex = false
				 * @param index The index to get
				 * @param isIndex If true returns the control at index else returns the control with id of index
				 * @retval IControl* The control or null on error / not finiding it etc
				 */
				virtual IControl *getControlAtIndex(const long index, bool isIndex) = 0;

				/**
				 * Get the index of a control
				 * @param control The control to find the index of
				 * @retval long The index of the control
				 */
				virtual long getControlIndex(IControl *control) = 0;

				/**
				 * Get the number of controls embedded in this root
				 * @retval long The number of controls
				 */
				virtual long getNumberOfControls() = 0;

				/**
				 * Get the number of controls embedded in this root, actually inserted <= getNumberOfControls
				 * @retval long The array insert index
				 */
				virtual long getNumberOfInsertedControls() = 0;

//	===========================================================================

				/**
				 * Draw the root control
				 * @param graphics The graphics context
				 */
				virtual void drawRootControl(CGraphics &graphics) = 0;

				/**
				 * Update (redraw) a control
				 * @param control The control to redraw
				 */
				virtual void updateControl(IControl *control) = 0;

				/**
				 * Update (redraw) a control at a specific index, or with a specific id if isIndex = false
				 * @param index The index ot update
				 * @param isIndex If true index is the controls id, else its the direct array index
				 */
				virtual void updateControl(const long index, bool isIndex) = 0;

				/**
				 * Update an area
				 * @param area The area to update
				 */
				virtual void updateArea(const CRect &area) = 0;

				/**
				 * Get the global location of the control in screen co-ordinates
				 * @param control The control to find the location of
				 * @param point The position of the control
				 */
				virtual void getGlobalCoordinatesOfControl(IControl *control, CPoint &point) = 0;

				/**
				 * Get the window (absoloute) position of a control
				 * @param control The control to find the absolute position of
				 * @param point The position of the control on return
				 */
				virtual void getWindowCoordinatesOfControl(IControl *control, CPoint &point) = 0;

//	===========================================================================

				/**
				 * Set the background image
				 * @param image The background image for the control root
				 */
				virtual void setBackgroundImage(IImage *image) = 0;

//	===========================================================================

				/**
				 * Start the timer to work out if the tool tip should fire up
				 */
				virtual void startToolTipTimer() = 0;

				/**
				 * Stop the tool tip timer from running
				 */
				virtual void stopToolTipTimer()  = 0;

//	===========================================================================

				// Save or load a layout of this panel..
				//virtual void serialise(const EControlRootSerialiseOption option);

//	===========================================================================

			};
		}
	}
}
#endif	// End of IControlRoot.hpp